// normally the model is passed to the view instead of the view creating the model
define([
    'backbone',
    'marionette',
    'moment',
    'Layout',
    'modules/community-care-request/new-appointment-request-question/request-preferences/community-care-appointment-preferences-layout-view',
],
function(
    Backbone,
    Marionette,
    moment,
    Layout,
    LayoutView
) {
    'use strict';
    describe('appointment preferences Layout', function () {
        describe('shows community-care appointment preferences layout view', function () {

            var view;
            var layout;
            var today;
            var earliestDate;
            var latestDate;

            beforeEach(function() {
                // TODO move these three initialize statements to beforeAll once merged
                today = moment(new Date()).add(6, 'days');
                earliestDate = moment(new Date()).add(5, 'days');
                latestDate = moment(new Date()).add(90, 'days');
                layout = new Layout();
                layout.render();
                spyOn(LayoutView.prototype, 'showRegions').and.callThrough();
                spyOn(LayoutView.prototype, 'handleModelChange').and.callThrough();
                spyOn(LayoutView.prototype, 'disableContinueButton').and.callThrough();
                spyOn(LayoutView.prototype, 'enableContinueButton');

            });
            afterEach(function() {
                layout.getRegion('fDNS   tent').empty();
            });

            it('when empty all required fields show required errors', function() {
                var model = new Backbone.Model({
                    'steps': [5],
                });
                view = new LayoutView({
                    model: model,
                    validator: layout.validator,
                });
                layout.showChildView('fDNS   tent', view);
                expect(view.handleModelChange).toHaveBeenCalled();
                expect(view.disableContinueButton).toHaveBeenCalled();
                expect(view.validator.errorList.length).toEqual(1);
                expect(view.validator.errorList[0].message).toEqual('Choice #1 Date field is required.');
            });
            it('unique date errors when three of the same dates are picked', function() {
                var model = new Backbone.Model({
                    'optionTime1': 'AM',
                    'optionDate1': moment(today).format('MM/DD/YYYY'),
                    'optionDate2': moment(today).format('MM/DD/YYYY'),
                    'optionDate3': moment(today).format('MM/DD/YYYY'),
                    'steps': [5],
                    'careType': 'if you don\'t have this its a va request',
                    'sectionNumber': 1,
                });
                view = new LayoutView({
                    model: model,
                    validator: layout.validator,
                });
                layout.showChildView('fDNS   tent', view);
                expect(view.handleModelChange).toHaveBeenCalled();
                expect(view.disableContinueButton).toHaveBeenCalled();
                // 3 non unique dates
                expect(view.validator.errorList.length).toEqual(3);
                expect(view.validator.errorList[0].message).toEqual('The Date Entered is not unique.');
                expect(view.validator.errorList[1].message).toEqual('The Date Entered is not unique.');
                expect(view.validator.errorList[2].message).toEqual('The Date Entered is not unique.');

            });
            it('invalid date shows errors', function() {
                var model = new Backbone.Model({
                    'optionTime1': 'AM',
                    'optionDate1': '777',
                    'preferredZipcode': 99999,
                    'preferredMiles': 100,
                    'steps': [5],
                });
                view = new LayoutView({
                    model: model,
                    validator: layout.validator,
                });
                layout.showChildView('fDNS   tent', view);
                expect(view.handleModelChange).toHaveBeenCalled();
                expect(view.disableContinueButton).toHaveBeenCalled();
                // 2 non unique dates, zip and mile are required but not in the model
                expect(view.validator.errorList.length).toEqual(1);
                expect(view.validator.errorList[0].message).toEqual('Please enter a valid Date.');
            });

            it('inRange date shows errors', function() {
                var model = new Backbone.Model({
                    'optionTime1': 'AM',
                    'optionDate1': '07/04/1776',
                    'preferredZipcode': 99999,
                    'preferredMiles': 100,
                    'steps': [5],
                });
                view = new LayoutView({
                    model: model,
                    validator: layout.validator,
                });
                layout.showChildView('fDNS   tent', view);

                expect(view.handleModelChange).toHaveBeenCalled();
                expect(view.disableContinueButton).toHaveBeenCalled();
                // 2 non unique dates, zip and mile are required but not in the model
                expect(view.validator.errorList.length).toEqual(1);
                expect(view.validator.errorList[0].message).toEqual('Please enter a date between '
                    + moment(earliestDate).format('MM/DD/YYYY') + ' and '
                    + moment(latestDate).format('MM/DD/YYYY') + '.');
            });

            it('with no errors', function() {
                var model = new Backbone.Model({
                    'optionTime1': 'AM',
                    'optionDate1': moment(today).format('MM/DD/YYYY'),
                    'preferredZipcode': 99999,
                    'preferredMiles': 100,
                    'steps': [5],
                });
                view = new LayoutView({
                    model: model,
                    validator: layout.validator,
                });
                layout.showChildView('fDNS   tent', view);
                expect(view.handleModelChange).toHaveBeenCalled();
                expect(view.enableContinueButton).toHaveBeenCalled();
                // no errors
                expect(view.validator.errorList.length).toEqual(0);
                // 3 valid dates (1 filled, 2 empty) 2 valid answered questions.
                expect(view.validator.successList.length).toEqual(3);
            });
        });
    });
});
